@extends('layouts.app')

@section('content')
<div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
    <!-- Motor Status Card -->
    <div class="bg-white rounded-xl shadow-lg p-6">
        <div class="flex items-center justify-between mb-4">
            <h2 class="text-xl font-bold text-gray-800">
                <i class="fas fa-cog text-blue-500 mr-2"></i>Water Pump Status
            </h2>
            <span id="motor-status" class="px-3 py-1 rounded-full text-sm font-semibold 
                {{ $motor->status == 'on' ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800' }}">
                {{ strtoupper($motor->status) }}
            </span>
        </div>
        
        <div class="mb-4">
            <p class="text-gray-600">
                <i class="far fa-clock mr-2"></i>
                Last Activated: 
                <span id="last-activated" class="font-semibold">
                    {{ $motor->last_activated_at ? $motor->last_activated_at->format('Y-m-d H:i:s') : 'Never' }}
                </span>
            </p>
        </div>
        
        <button id="toggle-motor" 
            class="w-full py-3 rounded-lg font-semibold transition duration-300
                {{ $motor->status == 'on' ? 'bg-red-500 hover:bg-red-600' : 'bg-green-500 hover:bg-green-600' }} 
                text-white">
            <i class="fas fa-power-off mr-2"></i>
            Turn {{ $motor->status == 'on' ? 'OFF' : 'ON' }} Water Pump
        </button>
    </div>

    <!-- Temperature Card -->
    <div class="bg-white rounded-xl shadow-lg p-6">
        <div class="flex items-center justify-between mb-4">
            <h2 class="text-xl font-bold text-gray-800">
                <i class="fas fa-thermometer-half text-red-500 mr-2"></i>Temperature
            </h2>
            <i class="fas fa-temperature-high text-red-500 text-2xl"></i>
        </div>
        
        <div class="text-center">
            <p class="text-5xl font-bold text-gray-800 mb-2" id="current-temperature">
                {{ $latestSensorData->temperature ?? '0' }}°
            </p>
            <p class="text-gray-600">Celsius</p>
        </div>
        
        <div class="mt-4 text-sm text-gray-500">
            <i class="far fa-clock mr-1"></i>
            Updated: <span id="temp-updated">{{ $latestSensorData->updated_at->format('H:i:s') ?? '--:--:--' }}</span>
        </div>
    </div>

    <!-- Humidity Card -->
    <div class="bg-white rounded-xl shadow-lg p-6">
        <div class="flex items-center justify-between mb-4">
            <h2 class="text-xl font-bold text-gray-800">
                <i class="fas fa-tint text-blue-500 mr-2"></i>Humidity
            </h2>
            <i class="fas fa-water text-blue-500 text-2xl"></i>
        </div>
        
        <div class="text-center">
            <p class="text-5xl font-bold text-gray-800 mb-2" id="current-humidity">
                {{ $latestSensorData->humidity ?? '0' }}
            </p>
            <p class="text-gray-600">Percentage</p>
        </div>
        
        <div class="mt-4">
            <div class="bg-gray-200 rounded-full h-2">
                <div class="bg-blue-500 rounded-full h-2" 
                     style="width: {{ min($latestSensorData->humidity ?? 0, 100) }}%"></div>
            </div>
        </div>
        
        <div class="mt-4 text-sm text-gray-500">
            <i class="far fa-clock mr-1"></i>
            Updated: <span id="humidity-updated">{{ $latestSensorData->updated_at->format('H:i:s') ?? '--:--:--' }}</span>
        </div>
    </div>

    <!-- Soil Moisture Card -->
    <div class="bg-white rounded-xl shadow-lg p-6 col-span-1 md:col-span-2 lg:col-span-3">
        <div class="flex items-center justify-between mb-6">
            <h2 class="text-xl font-bold text-gray-800">
                <i class="fas fa-seedling text-green-500 mr-2"></i>Soil Moisture
            </h2>
            <div class="flex items-center">
                <span id="soil-status" class="px-3 py-1 rounded-full text-sm font-semibold mr-3
                    {{ ($latestSensorData->soil_moisture ?? 0) > 500 ? 'bg-yellow-100 text-yellow-800' : 'bg-green-100 text-green-800' }}">
                    {{ ($latestSensorData->soil_moisture ?? 0) > 500 ? 'DRY' : 'WET' }}
                </span>
                <i class="fas fa-cloud-rain text-green-500 text-2xl"></i>
            </div>
        </div>
        
        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
            <div class="text-center">
                <p class="text-6xl font-bold text-gray-800 mb-2" id="current-soil-moisture">
                    {{ $latestSensorData->soil_moisture ?? '0' }}
                </p>
                <p class="text-gray-600">Sensor Value (0-1023)</p>
                <div class="mt-4 text-sm text-gray-500">
                    <i class="far fa-clock mr-1"></i>
                    Updated: <span id="soil-updated">{{ $latestSensorData->updated_at->format('H:i:s') ?? '--:--:--' }}</span>
                </div>
            </div>
            
            <div class="flex flex-col justify-center">
                <div class="mb-4">
                    <p class="text-gray-700 mb-2">Moisture Level:</p>
                    <div class="bg-gray-200 rounded-full h-6 overflow-hidden">
                        <div class="bg-gradient-to-r from-green-500 via-yellow-500 to-red-500 h-full" 
                             style="width: {{ min(($latestSensorData->soil_moisture ?? 0) / 10.23, 100) }}%"></div>
                    </div>
                    <div class="flex justify-between text-xs text-gray-500 mt-1">
                        <span>Wet (0)</span>
                        <span>Optimal (512)</span>
                        <span>Dry (1023)</span>
                    </div>
                </div>
                
                <div class="bg-gray-50 p-4 rounded-lg">
                    <h4 class="font-semibold text-gray-800 mb-2">
                        <i class="fas fa-info-circle text-blue-500 mr-2"></i>Interpretation
                    </h4>
                    <p class="text-sm text-gray-600" id="soil-interpretation">
                        @if(($latestSensorData->soil_moisture ?? 0) < 300)
                            Soil is very wet. Irrigation not needed.
                        @elseif(($latestSensorData->soil_moisture ?? 0) < 600)
                            Soil moisture is optimal.
                        @else
                            Soil is dry. Consider irrigation.
                        @endif
                    </p>
                </div>
            </div>
        </div>
    </div>
</div>

@push('scripts')
<script>
$(document).ready(function() {
    // Toggle motor status
    $('#toggle-motor').click(function() {
        const currentStatus = "{{ $motor->status }}";
        const newStatus = currentStatus === 'on' ? 'off' : 'on';
        
        $.ajax({
            url: "{{ route('motor.update') }}",
            method: 'POST',
            data: {
                _token: "{{ csrf_token() }}",
                status: newStatus
            },
            success: function(response) {
                $('#motor-status').text(newStatus.toUpperCase());
                if (newStatus === 'on') {
                    $('#motor-status').removeClass('bg-red-100 text-red-800')
                        .addClass('bg-green-100 text-green-800');
                    $('#toggle-motor')
                        .removeClass('bg-green-500 hover:bg-green-600')
                        .addClass('bg-red-500 hover:bg-red-600')
                        .html('<i class="fas fa-power-off mr-2"></i>Turn OFF Water Pump');
                } else {
                    $('#motor-status').removeClass('bg-green-100 text-green-800')
                        .addClass('bg-red-100 text-red-800');
                    $('#toggle-motor')
                        .removeClass('bg-red-500 hover:bg-red-600')
                        .addClass('bg-green-500 hover:bg-green-600')
                        .html('<i class="fas fa-power-off mr-2"></i>Turn ON Water Pump');
                }
                
                if (response.last_activated) {
                    $('#last-activated').text(response.last_activated);
                }
            }
        });
    });

    // Update sensor data every 10 seconds
    function updateSensorData() {
        $.get("{{ route('sensors.latest') }}", function(data) {
            $('#current-temperature').text(data.temperature + '°');
            $('#current-humidity').text(data.humidity);
            $('#current-soil-moisture').text(data.soil_moisture);
            
            // Update timestamps
            const time = new Date(data.updated_at).toLocaleTimeString();
            $('#temp-updated, #humidity-updated, #soil-updated').text(time);
            
            // Update soil status
            if (data.soil_moisture > 500) {
                $('#soil-status').text('DRY')
                    .removeClass('bg-green-100 text-green-800')
                    .addClass('bg-yellow-100 text-yellow-800');
            } else {
                $('#soil-status').text('WET')
                    .removeClass('bg-yellow-100 text-yellow-800')
                    .addClass('bg-green-100 text-green-800');
            }
            
            // Update soil interpretation
            let interpretation = '';
            if (data.soil_moisture < 300) {
                interpretation = 'Soil is very wet. Irrigation not needed.';
            } else if (data.soil_moisture < 600) {
                interpretation = 'Soil moisture is optimal.';
            } else {
                interpretation = 'Soil is dry. Consider irrigation.';
            }
            $('#soil-interpretation').text(interpretation);
        });
    }

    // Initial update
    updateSensorData();
    
    // Set interval for updates
    setInterval(updateSensorData, 10000);
});
</script>
@endpush
@endsection